# Beyti API - Complete Summary

## Project Overview

Beyti Visitor Management System API - A comprehensive Laravel-based backend for managing visitor access requests and approvals.

## Recent Updates

### 1. ✅ Dashboard Endpoints

#### `/api/dashboard/visit-status`
**Purpose**: Get visit counts grouped by status with date range filtering

**Query Parameters**:
- `range` (required): `today`, `yesterday`, `last_7_days`, `last_30_days`

**Response**:
```json
{
  "completed": 6,
  "approved": 5,
  "pending_approve": 2,
  "pending_visitor": 2,
  "expired": 1,
  "rejected": 2,
  "canceled": 1
}
```

**Features**:
- Handles both single-day and multi-day visits correctly
- Single meetings: Checks if date equals target date
- Multi-day meetings: Checks if target date is within date range

#### `/api/dashboard/today-visits`
**Purpose**: Get all visits scheduled for today

**Features**:
- Returns both single-day visits (date = today) and multi-day visits (today is between date_from and date_to)
- Includes location and visitor information
- Formatted dates with times

#### `/api/dashboard/recent-visits`
**Purpose**: Get most recent visits with online/offline status

**Query Parameters**:
- `limit` (optional, default: 5): Number of visits to return

**Features**:
- Online status: Visit date matches today
- Sorted by creation date (most recent first)

### 2. ✅ Visit Management Endpoints

#### `DELETE /api/visits/{id}` - Cancel Visit
**Purpose**: Soft delete a visit by changing status to "Canceled"

**Authorization**:
- Admin (can cancel any visit)
- Host (can only cancel their own visits)

**Status Validation**:
- ✅ Can cancel: Pending Visitor, Pending Approve, Pending Host, Approved
- ❌ Cannot cancel: Completed, Rejected, Expired, Canceled

**Response**:
```json
{
  "message": "Visit canceled successfully",
  "status": "Canceled"
}
```

#### `POST /api/visits/{id}/return` - Return to Host
**Purpose**: Return visit to host for corrections (security role)

**Authorization**: Security Member, Security Admin, or Admin

**Request Body**:
```json
{
  "reason": "Please provide additional documentation"
}
```

**Validation**:
- `reason`: Required, min 10 chars, max 500 chars

**Status Change**: Pending Approve → Pending Host

**Response**:
```json
{
  "message": "Visit returned to host for corrections",
  "status": "Pending Host",
  "data": {
    "id": 1,
    "status": "Pending Host",
    "rejection_reason": "Please provide additional documentation"
  }
}
```

#### `POST /api/visits/{id}/reject` - Reject Visit
**Purpose**: Reject a visit request (security role)

**Authorization**: Security Member, Security Admin, or Admin

**Request Body**:
```json
{
  "reason": "Security concerns regarding visitor background"
}
```

**Validation**:
- `reason`: Required, min 10 chars

**Status Change**: Pending Approve → Rejected

#### `POST /api/visits/{id}/approve` - Approve Visit
**Purpose**: Approve a visit request (security role)

**Authorization**: Security Member, Security Admin, or Admin

**Status Change**: Pending Approve → Approved

#### `GET /api/visits/{id}/tracking-history` - Get Status Tracking History ✨ NEW
**Purpose**: Retrieve complete status change history for a visit (audit trail)

**Authorization**:
- Admin (can view any visit history)
- Security (can view any visit history)
- Host (can only view their own visit history)

**Response**:
```json
{
  "visit_id": 1,
  "request_id": "VIS-20260111-0001",
  "current_status": "Approved",
  "history": [
    {
      "id": 1,
      "from_status": null,
      "to_status": "Pending Visitor",
      "changed_by_email": "host@beyti.test",
      "changed_by_name": "Omar Mahmoud",
      "reason": "Visit created",
      "changed_at": "2026-01-10 10:30:00",
      "changed_at_human": "2 hours ago"
    },
    {
      "id": 2,
      "from_status": "Pending Visitor",
      "to_status": "Pending Approve",
      "changed_by_email": null,
      "changed_by_name": null,
      "reason": "Visitor registered",
      "changed_at": "2026-01-10 11:00:00",
      "changed_at_human": "1 hour ago"
    },
    {
      "id": 3,
      "from_status": "Pending Approve",
      "to_status": "Approved",
      "changed_by_email": "security@beyti.test",
      "changed_by_name": "Fatima Ali",
      "reason": "Visit approved by security",
      "changed_at": "2026-01-10 12:00:00",
      "changed_at_human": "30 minutes ago"
    }
  ]
}
```

**Features**:
- Chronological order (oldest to newest)
- Shows who made each change
- Includes reason for each transition
- Human-readable timestamps
- Complete audit trail

**Use Cases**:
- Compliance and auditing
- Troubleshooting status issues
- Understanding visit lifecycle
- Tracking accountability

### 3. ✅ Configuration Updates

#### Timezone Configuration
**File**: `config/app.php`, `.env`

**Setting**: `APP_TIMEZONE=Africa/Cairo`

**Impact**:
- All date/time operations now use Cairo timezone (UTC+2)
- `today()` and `now()` return correct local times
- Dashboard date queries work correctly

### 4. ✅ Comprehensive Test Data Seeder

#### `ComprehensiveVisitsSeeder`
**Purpose**: Generate realistic test data for all scenarios

**Data Generated**:
- **20 visits** with complete information
- **25 visitors** (some visits have multiple visitors)
- **80+ status history records**
- All 8 visit statuses covered
- Both meeting types (single & multiple day)
- Various date scenarios (today, yesterday, future, past)

**Status Distribution**:
| Status | Count |
|--------|-------|
| Completed | 6 |
| Approved | 5 |
| Pending Approve | 2 |
| Pending Visitor | 2 |
| Rejected | 2 |
| Pending Host | 1 |
| Canceled | 1 |
| Expired | 1 |

**Usage**:
```bash
php artisan db:seed --class=ComprehensiveVisitsSeeder
```

## API Endpoints Overview

### Authentication
- `POST /api/test/token` - Generate test token (dev only)
- `GET /api/test/users` - List test users (dev only)
- `GET /api/test/me` - Get current user info (dev only)

### Visits
- `GET /api/visits` - List all visits (with filters)
- `GET /api/visits/{id}` - Get visit details
- `POST /api/visits` - Create new visit
- `PUT /api/visits/{id}` - Update visit
- `DELETE /api/visits/{id}` - Cancel visit ✨ NEW
- `POST /api/visits/{id}/approve` - Approve visit
- `POST /api/visits/{id}/reject` - Reject visit
- `POST /api/visits/{id}/return` - Return to host ✨ DOCUMENTED
- `POST /api/visits/{id}/send-reminder` - Send reminder email
- `GET /api/visits/{id}/tracking-history` - Get status tracking history ✨ NEW

### Dashboard
- `GET /api/dashboard/stats` - Get statistics
- `GET /api/dashboard/recent-visits` - Get recent visits
- `GET /api/dashboard/today-visits` - Get today's visits
- `GET /api/dashboard/visit-status` - Get status counts ✨ NEW

### Locations
- `GET /api/locations` - Get all locations

### Settings (Admin only)
- `GET /api/settings` - Get all settings
- `PUT /api/settings/azure-ad` - Update Azure AD config
- `PUT /api/settings/dauha` - Update Dauha DSS config
- `PUT /api/settings/smtp` - Update SMTP config
- `PUT /api/settings/timezone` - Update timezone

## Visit Status Workflow

```
[Host Creates Visit]
        ↓
  Pending Visitor ──────────┐
        ↓                   │
[Visitor Registers]         │ DELETE (Cancel)
        ↓                   │
  Pending Approve ──────┐   │
        ↓               │   │
   ┌────┴────┐          │   │
   ↓         ↓          │   │
Approved  Reject    Return  │
   ↓         ↓          ↓   │
Completed Rejected  Pending Host
                           │
                      [Host Fixes]
                           ↓
                    Pending Approve
```

### Status Descriptions

| Status | Description | Can Cancel? | Next Actions |
|--------|-------------|-------------|--------------|
| Pending Visitor | Awaiting visitor registration | ✅ Yes | Register, Cancel |
| Pending Approve | Awaiting security approval | ✅ Yes | Approve, Reject, Return, Cancel |
| Pending Host | Returned to host for corrections | ✅ Yes | Update & Resubmit, Cancel |
| Approved | Approved by security | ✅ Yes | Mark Complete, Cancel |
| Completed | Visit finished | ❌ No | None (final) |
| Rejected | Denied by security | ❌ No | None (final) |
| Expired | Registration token expired | ❌ No | None (final) |
| Canceled | Canceled by host/admin | ❌ No | None (final) |

## Test Users

| Email | Role | Name | Password (N/A for API) |
|-------|------|------|------------------------|
| admin@beyti.test | admin | Ahmed Mohamed | - |
| security.admin@beyti.test | security_admin | Khaled Hassan | - |
| security@beyti.test | security_member | Fatima Ali | - |
| host@beyti.test | user | Omar Mahmoud | - |

## Postman Collection

### Updated Endpoints
1. ✅ **Get Visit Status Counts** - Dashboard section
2. ✅ **Delete Visit (Cancel)** - Visits section
3. ✅ **Return Visit to Host** - Visits section
4. ✅ **Reject Visit** - Visits section (already existed)

### Collection File
`Beyti-API.postman_collection.json`

### Environment Variables
- `baseUrl`: `http://localhost:8080/beyti-api/public`
- `authToken`: Auto-saved from "Generate Test Token" request

## Documentation Files

| File | Purpose |
|------|---------|
| `README-SETUP.md` | Setup instructions |
| `TEST-AUTH-GUIDE.md` | Authentication testing guide |
| `POSTMAN-GUIDE.md` | Postman usage guide |
| `SEEDER-GUIDE.md` | Comprehensive seeder documentation ✨ NEW |
| `RETURN-ENDPOINT-GUIDE.md` | Return visit endpoint guide ✨ NEW |
| `DELETE-ENDPOINT-GUIDE.md` | Delete visit endpoint guide ✨ NEW |
| `API-SUMMARY.md` | This file ✨ NEW |

## Database Schema

### Key Tables
- `visits` - Visit requests
- `visitors` - Visitor information
- `visit_status_history` - Status change audit trail
- `user_roles` - User authentication and roles
- `locations` - Visit locations
- `settings` - System configuration
- `email_templates` - Email notification templates

## Quick Start Testing

### 1. Generate Test Token
```bash
POST /api/test/token
{
  "email": "admin@beyti.test"
}
```
Token is auto-saved to `{{authToken}}` in Postman.

### 2. Run Seeder (Optional - for fresh data)
```bash
php artisan migrate:fresh --seed
```

### 3. Test Dashboard
```bash
GET /api/dashboard/visit-status?range=today
GET /api/dashboard/today-visits
GET /api/dashboard/recent-visits?limit=10
```

### 4. Test Visit Operations
```bash
# Get all visits
GET /api/visits

# Get specific visit
GET /api/visits/1

# Cancel a visit (ID 3 is Pending Visitor in test data)
DELETE /api/visits/3

# Return visit to host (ID 8 is Pending Approve)
POST /api/visits/8/return
{
  "reason": "Please provide company registration certificate"
}

# Reject visit (ID 19 is Pending Approve)
POST /api/visits/19/reject
{
  "reason": "Security concerns regarding visitor background"
}
```

## Development Tips

### Clear Cache After Changes
```bash
php artisan config:clear
php artisan cache:clear
php artisan route:clear
```

### Check Routes
```bash
php artisan route:list
php artisan route:list | grep visits
php artisan route:list | grep dashboard
```

### Database Queries for Testing
```bash
# Count visits by status
php artisan tinker --execute="DB::table('visits')->select('status', DB::raw('count(*) as count'))->groupBy('status')->get();"

# Get today's visits
php artisan tinker --execute="echo Visit::whereDate('date', today())->count();"

# Find visits that can be deleted
php artisan tinker --execute="echo Visit::whereIn('status', ['Pending Visitor', 'Pending Approve', 'Pending Host', 'Approved'])->count();"
```

## Recent Changes Summary

### What's New in This Update

1. **Dashboard Enhancements**
   - New `/api/dashboard/visit-status` endpoint with date range filtering
   - Fixed date logic to properly handle multi-day visits
   - Updated timezone to Africa/Cairo (UTC+2)

2. **Visit Management**
   - Delete endpoint now properly validates status
   - Return endpoint documented in Postman
   - All status transitions properly tracked

3. **Testing Infrastructure**
   - Comprehensive seeder with 20 realistic visits
   - All 8 statuses represented in test data
   - Status history properly generated

4. **Documentation**
   - Complete Postman collection with all endpoints
   - Individual guides for complex endpoints
   - This summary document for quick reference

### Bug Fixes
- ✅ Fixed timezone issues (UTC → Africa/Cairo)
- ✅ Fixed today's visits query to include multi-day visits
- ✅ Fixed delete endpoint variable reference issues
- ✅ Updated canBeDeleted() validation logic

### Breaking Changes
None - All changes are backward compatible.

## Next Steps / Future Enhancements

Potential areas for future development:
1. Email notifications for cancellations
2. Bulk operations (approve/reject multiple visits)
3. Visit statistics and reporting
4. Export functionality (PDF, Excel)
5. Real-time notifications (WebSockets)
6. File upload for visitor documents
7. QR code generation for approved visits
8. Visitor check-in/check-out tracking

## Support & Troubleshooting

### Common Issues

1. **"Forbidden" errors**
   - Solution: Check user role and permissions
   - Generate token with correct user (admin, security, or host)

2. **"Cannot cancel visit" errors**
   - Solution: Check visit status - only certain statuses can be canceled
   - Final statuses (Completed, Rejected, Expired, Canceled) cannot be changed

3. **Empty dashboard results**
   - Solution: Run seeder to generate test data
   - Check timezone configuration

4. **Date/time inconsistencies**
   - Solution: Ensure APP_TIMEZONE is set to Africa/Cairo
   - Clear config cache after changes

### Getting Help

For issues or questions:
1. Check the specific endpoint guide (e.g., DELETE-ENDPOINT-GUIDE.md)
2. Review Postman collection examples
3. Check seeder data with database queries
4. Review API-SUMMARY.md (this file)

## Version Information

- **Laravel Version**: 11.x
- **PHP Version**: 8.2+
- **Database**: MySQL
- **API Version**: v1
- **Last Updated**: January 6, 2026

---

**Note**: This is a development/testing API. Production deployment requires additional security hardening, proper authentication (Azure AD integration), and environment-specific configuration.
