# Comprehensive Visits Seeder Guide

## Overview

The `ComprehensiveVisitsSeeder` creates **20 realistic test visits** covering all statuses, meeting types, and date scenarios to thoroughly test the Beyti API endpoints.

## Quick Start

```bash
# Clear existing visits and run the seeder
php artisan tinker --execute="DB::statement('SET FOREIGN_KEY_CHECKS=0;'); DB::table('visit_status_history')->delete(); DB::table('visitors')->delete(); DB::table('visits')->delete(); DB::statement('SET FOREIGN_KEY_CHECKS=1;'); echo 'Tables cleared';"

php artisan db:seed --class=ComprehensiveVisitsSeeder
```

Or run all seeders at once:

```bash
php artisan migrate:fresh --seed
```

## What Gets Created

### 📊 Total Test Data
- **20 visits** with complete data
- **25 visitors** (including additional visitors for some visits)
- **Full status history** for each visit tracking all status changes
- All **meeting types** (single and multiple day)
- All **8 visit statuses**

### 📅 Date Distribution

#### Today (4 visits)
- ✅ **Approved** - Single day meeting (John Doe from Tech Corp)
- ✅ **Approved** - Multiple day spanning today (Sarah Wilson from Wilson Consulting)
- 📧 **Pending Visitor** - Awaiting registration (Mike Brown)
- ✔️ **Completed** - Early morning meeting (Emma Davis from Suppliers Inc)

#### Yesterday (3 visits)
- ✔️ **Completed** - Yesterday's meeting (Alex Johnson from Partners LLC)
- ❌ **Rejected** - Security rejection (suspicious@unknown.com)
- ✅ **Approved** - Multiple day meeting that included yesterday

#### Tomorrow (2 visits)
- ✅ **Approved** - Scheduled for tomorrow (Lisa Anderson)
- ⏳ **Pending Approve** - Awaiting security approval (Robert Taylor)

#### Last 7 Days (9 total visits including today's)
- ✔️ Completed meetings from 3 and 5 days ago
- ⏰ **Expired** - Token expired 6 days ago

#### Last 30 Days (20 total visits)
- ✔️ Completed meetings from 10, 15-12 days ago (including multi-day)
- ❌ **Canceled** - 20 days ago
- ❌ **Rejected** - 25 days ago

#### Future (4 visits)
- ✅ **Approved** - 3 days from now (Kevin Park)
- 📧 **Pending Visitor** - 5 days from now
- ✅ **Approved** - Multi-day 7-10 days from now (Michelle Chen)
- ⏳ **Pending Approve** - 7 days from now (Thomas Anderson)
- 🔄 **Pending Host** - 4 days from now, returned for clarification (Rachel Green)

### 🏷️ Status Distribution

| Status | Count | Description |
|--------|-------|-------------|
| ✔️ Completed | 6 | Successfully finished visits |
| ✅ Approved | 5 | Approved by security, ready for visit |
| ⏳ Pending Approve | 2 | Visitor registered, awaiting security review |
| 📧 Pending Visitor | 2 | Invitation sent, awaiting visitor registration |
| ❌ Rejected | 2 | Rejected by security team |
| 🔄 Pending Host | 1 | Returned to host for additional info |
| ❌ Canceled | 1 | Canceled visit |
| ⏰ Expired | 1 | Registration token expired |

## Test Users

The seeder uses these test users as hosts:

| Email | Name | Role | Phone |
|-------|------|------|-------|
| admin@beyti.test | Ahmed Mohamed | Admin | 01021002800 |
| security.admin@beyti.test | Khaled Hassan | Security Admin | 01012345678 |
| security@beyti.test | Fatima Ali | Security Member | 01098765432 |
| host@beyti.test | Omar Mahmoud | User | 01055556666 |

## Testing Endpoints

### 1. Visit Status Counts

Test all date ranges:

```bash
# Today
GET /api/dashboard/visit-status?range=today
# Expected: {"completed": 1, "approved": 2, "pending_visitor": 1, ...}

# Yesterday
GET /api/dashboard/visit-status?range=yesterday
# Expected: {"completed": 1, "approved": 1, "rejected": 1, ...}

# Last 7 days
GET /api/dashboard/visit-status?range=last_7_days
# Expected: {"completed": 4, "approved": 2, "expired": 1, ...}

# Last 30 days (all visits)
GET /api/dashboard/visit-status?range=last_30_days
# Expected: {"completed": 6, "approved": 5, "pending_approve": 2, ...}
```

### 2. Today's Visits

```bash
GET /api/dashboard/today-visits
# Returns: 4 visits scheduled for today (both single and multi-day)
```

### 3. Recent Visits

```bash
GET /api/dashboard/recent-visits?limit=10
# Returns: 10 most recent visits with online/offline status
```

### 4. Get All Visits with Filters

```bash
# Filter by status
GET /api/visits?status=Approved
# Expected: 5 approved visits

# Filter by date range
GET /api/visits?date_from=2026-01-06&date_to=2026-01-06
# Expected: Today's visits
```

## Data Quality Features

### ✨ Realistic Data
- Proper company names and email addresses
- Detailed visit reasons (min 10 chars requirement met)
- Realistic phone numbers (Egyptian format)
- Business hours compliance (8 AM - 6 PM)

### 🔗 Complete Relationships
- Main visitors with full registration details
- Additional visitors for some visits (5 visits have 2 visitors each)
- Status history tracking all transitions
- Foreign key relationships properly maintained

### 📋 Status Flow Accuracy
- **Pending Visitor** → Registration link sent
- **Pending Approve** → Visitor registered, needs security review
- **Pending Host** → Returned with specific reasons
- **Approved** → Full approval chain recorded
- **Rejected** → Rejection reasons included
- **Completed** → Full lifecycle tracked
- **Canceled** → Cancellation recorded
- **Expired** → Token expiration tracked

### 🗂️ Meeting Type Coverage
- **Single day meetings**: Date, start_time, end_time
- **Multiple day meetings**: date_from, date_to, spanning multiple days
- Proper date overlap logic for "today" queries

## Cleanup

To reset and reseed:

```bash
# Option 1: Clear visits only
php artisan tinker --execute="DB::statement('SET FOREIGN_KEY_CHECKS=0;'); DB::table('visit_status_history')->delete(); DB::table('visitors')->delete(); DB::table('visits')->delete(); DB::statement('SET FOREIGN_KEY_CHECKS=1;'); echo 'Cleared';"
php artisan db:seed --class=ComprehensiveVisitsSeeder

# Option 2: Fresh migration (resets everything)
php artisan migrate:fresh --seed
```

## Verification

After seeding, verify the data:

```bash
# Check visit counts by status
php artisan tinker --execute="DB::table('visits')->select('status', DB::raw('count(*) as count'))->groupBy('status')->get();"

# Check today's visits
php artisan tinker --execute="DB::table('visits')->whereDate('date', today())->orWhere(function(\$q) { \$q->where('meeting_type', 'multiple')->whereDate('date_from', '<=', today())->whereDate('date_to', '>=', today()); })->count();"
```

## Expected Results

### `/api/dashboard/visit-status?range=today`
```json
{
  "completed": 1,
  "approved": 2,
  "pending_approve": 0,
  "pending_visitor": 1,
  "expired": 0,
  "rejected": 0,
  "canceled": 0
}
```

### `/api/dashboard/visit-status?range=last_30_days`
```json
{
  "completed": 6,
  "approved": 5,
  "pending_approve": 2,
  "pending_visitor": 2,
  "expired": 1,
  "rejected": 2,
  "canceled": 1
}
```

## Benefits

1. **Complete Coverage**: Tests all 8 statuses and all date scenarios
2. **Realistic**: Production-like data with proper relationships
3. **Repeatable**: Can be run multiple times with consistent results
4. **Fast**: Seeds 20+ visits in seconds
5. **Comprehensive**: Includes visitors, status history, and all edge cases

## Use Cases

- ✅ Testing dashboard statistics
- ✅ Validating date range filters
- ✅ Testing status workflows
- ✅ Demo/presentation data
- ✅ Frontend development
- ✅ API integration testing
- ✅ Performance testing with realistic data
