# Test Authentication Guide

This guide explains how to test the Beyti API endpoints without Azure AD integration during development.

## ⚠️ Important: Development Only

**These test authentication endpoints should be REMOVED before deploying to production!**

All test endpoints are under the `/api/test` prefix and should be deleted from:
- `routes/api.php` (lines 24-35)
- `app/Http/Controllers/Api/TestAuthController.php` (entire file)
- `database/seeders/TestUserSeeder.php` (entire file)

---

## Available Test Users

Four test users have been created with different roles:

| Email | Role | Permissions |
|-------|------|-------------|
| `admin@beyti.test` | admin | Full access to all endpoints including settings |
| `security.admin@beyti.test` | security_admin | Can approve/reject/return visits |
| `security@beyti.test` | security_member | Can approve/reject/return visits |
| `host@beyti.test` | user | Can create and manage visits |

---

## Quick Start with cURL

### 1. List Available Test Users

```bash
curl http://localhost:8080/beyti-api/public/api/test/users
```

Response:
```json
{
  "message": "Test users retrieved successfully",
  "users": [
    {"email": "admin@beyti.test", "role": "admin", "is_active": true},
    {"email": "security@beyti.test", "role": "security_member", "is_active": true}
  ]
}
```

### 2. Generate Authentication Token

```bash
curl -X POST http://localhost:8080/beyti-api/public/api/test/token \
  -H "Content-Type: application/json" \
  -d '{"email":"admin@beyti.test"}'
```

Response:
```json
{
  "message": "Token generated successfully",
  "access_token": "1|XcVASoEjkFNpyfMbYZJNFqMuu2ZwSpQFsOYzWHPJa75fb540",
  "token_type": "Bearer",
  "user": {
    "email": "admin@beyti.test",
    "role": "admin",
    "is_active": true
  }
}
```

### 3. Use Token to Access Protected Endpoints

```bash
curl http://localhost:8080/beyti-api/public/api/locations \
  -H "Authorization: Bearer 1|XcVASoEjkFNpyfMbYZJNFqMuu2ZwSpQFsOYzWHPJa75fb540"
```

Response:
```json
[
  {"id": 1, "name": "Cairo Main Office", "code": "CAI-001"},
  {"id": 2, "name": "Alexandria Branch", "code": "ALX-001"}
]
```

### 4. Verify Current User

```bash
curl http://localhost:8080/beyti-api/public/api/test/me \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 5. Revoke Token (Logout)

```bash
curl -X POST http://localhost:8080/beyti-api/public/api/test/revoke \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Using with Postman

### Step 1: Import Collection

The Postman collection has been updated with a "Test Authentication (Development Only)" folder containing all test endpoints.

### Step 2: Generate Token

1. Open "Test Authentication (Development Only)" folder
2. Click "Generate Test Token"
3. Change the email in the request body if needed (default: `admin@beyti.test`)
4. Click "Send"
5. **The token is automatically saved to the `authToken` environment variable!**

### Step 3: Test Protected Endpoints

Now all protected endpoints will work automatically using the saved token. Try:

- **Locations** → "Get All Locations"
- **Dashboard** → "Get Statistics"
- **Visits** → "Get All Visits"

### Step 4: Switch Users

To test with a different role:

1. Go to "Generate Test Token"
2. Change the email to another test user (e.g., `security@beyti.test`)
3. Send the request
4. The new token automatically replaces the old one

---

## Test Authentication Endpoints

All endpoints are prefixed with `/api/test`:

### GET /api/test/users
List all available test users.

**Response:**
```json
{
  "message": "Test users retrieved successfully",
  "users": [...],
  "note": "Use POST /api/test/token with any of these emails to get an access token"
}
```

---

### POST /api/test/token
Generate an authentication token for a test user.

**Request Body:**
```json
{
  "email": "admin@beyti.test"
}
```

**Response:**
```json
{
  "message": "Token generated successfully",
  "access_token": "1|...",
  "token_type": "Bearer",
  "user": {
    "email": "admin@beyti.test",
    "role": "admin",
    "is_active": true
  }
}
```

---

### GET /api/test/me
Get currently authenticated user information.

**Headers:**
```
Authorization: Bearer YOUR_TOKEN
```

**Response:**
```json
{
  "email": "admin@beyti.test",
  "role": "admin",
  "is_active": true,
  "last_login_at": "2025-12-31T15:23:06.000000Z"
}
```

---

### POST /api/test/revoke
Revoke (logout) the current token.

**Headers:**
```
Authorization: Bearer YOUR_TOKEN
```

**Response:**
```json
{
  "message": "Token revoked successfully"
}
```

---

## Testing Different Roles

### Admin User
```bash
# Generate admin token
curl -X POST http://localhost:8080/beyti-api/public/api/test/token \
  -H "Content-Type: application/json" \
  -d '{"email":"admin@beyti.test"}'

# Test admin-only endpoint (Settings)
curl http://localhost:8080/beyti-api/public/api/settings \
  -H "Authorization: Bearer ADMIN_TOKEN"
```

### Security Member
```bash
# Generate security member token
curl -X POST http://localhost:8080/beyti-api/public/api/test/token \
  -H "Content-Type: application/json" \
  -d '{"email":"security@beyti.test"}'

# Test visit approval (security members can approve)
curl -X POST http://localhost:8080/beyti-api/public/api/visits/1/approve \
  -H "Authorization: Bearer SECURITY_TOKEN"
```

### Host User
```bash
# Generate host token
curl -X POST http://localhost:8080/beyti-api/public/api/test/token \
  -H "Content-Type: application/json" \
  -d '{"email":"host@beyti.test"}'

# Test creating a visit (any authenticated user can create)
curl -X POST http://localhost:8080/beyti-api/public/api/visits \
  -H "Authorization: Bearer HOST_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{...}'
```

---

## Token Information

- **Token Type:** Bearer token
- **Expiration:** 30 days (43,200 minutes) - configured in `config/sanctum.php`
- **Storage:** Stored in `personal_access_tokens` table
- **Format:** `{token_id}|{plaintext_token}`

---

## Troubleshooting

### "Unauthenticated" Error

1. **Check token format:** Must be `Bearer {token}` in Authorization header
2. **Verify token hasn't expired:** Tokens expire after 30 days
3. **Ensure token hasn't been revoked:** Generate a new token
4. **Check user is active:** User must have `is_active = true`

### "Forbidden" Error

The authenticated user doesn't have the required role for this endpoint.

**Examples:**
- Settings endpoints require `admin` role
- Visit approval requires `security_member`, `security_admin`, or `admin` role

### Token Not Working After Database Changes

If you've reset the database or made changes to the `personal_access_tokens` table:

1. Generate a new token
2. Clear Laravel caches: `php artisan optimize:clear`

---

## Before Going to Production

**CRITICAL: Remove all test authentication code!**

### Files to Delete:
1. `app/Http/Controllers/Api/TestAuthController.php`
2. `database/seeders/TestUserSeeder.php`
3. `TEST-AUTH-GUIDE.md` (this file)

### Code to Remove:
1. In `routes/api.php`: Delete lines 24-35 (Test Authentication Routes section)
2. In `routes/api.php`: Remove the TestAuthController import (line 11)

### Clean Database:
```bash
# Remove test users
DELETE FROM user_roles WHERE email LIKE '%@beyti.test';

# Remove test tokens
DELETE FROM personal_access_tokens WHERE tokenable_id LIKE '%@beyti.test';
```

---

## Next Steps

Once Azure AD is integrated:

1. Remove test authentication endpoints
2. Update Postman collection to remove "Test Authentication" folder
3. Use Azure AD login flow for real authentication
4. Test with real Azure AD accounts

---

## Need Help?

- Check the main API documentation: `README-SETUP.md`
- Review Postman collection guide: `POSTMAN-GUIDE.md`
- Check Laravel logs: `storage/logs/laravel.log`
